/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpEnumeratedSet.c,v 1.17 2001/01/25 22:11:08 jeffc Exp $
____________________________________________________________________________*/
/*
 * pgpEnumeratedSet.c -- Arbitrary enumerated subset variant of pgpKeySet
 */
#include "pgpConfig.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <ctype.h>

#include "pgpKeyPriv.h"
#include "pgpDebug.h"
#include "pgpErrors.h"
#include "pgpTypes.h"
#include "pgpMem.h"
#include "pgpContext.h"


/* An entry representing an object in the set */
typedef struct ObjEntry
{
	PGPKeyDBObj		*obj;
	PGPInt32		numChildren;
} ObjEntry;
#define kAllChildren -1

/* Private data for KeySet of type: enumerated subset */
typedef struct EnumeratedSubsetPriv
{
	ObjEntry *		objArray;
	ObjEntry **		keyPointers;		/* Sorted by keyid */
	PGPUInt32		numKeys;
	PGPUInt32		numObjs;
} EnumeratedSubsetPriv;


static PGPError	NewEnumeratedSubsetInternal(PGPKeyDBRef		db,
											PGPKeySetRef *	newSet);


/*
 * Check to see if a (top level) key is a member of an enumerated subset
 * If so return ObjEntry pointer, else return NULL
 */
    static ObjEntry *
sFindKeyMemberOfEnumeratedSubset(
	PGPKeySetRef			set,
	PGPKeyDBObjRef			key)
{
	EnumeratedSubsetPriv *	priv = (EnumeratedSubsetPriv *)set->priv;
	PGPKeyID				keyID;
	PGPInt32				lowerBound;
	PGPInt32				upperBoundPlusOne;
	PGPInt32				indexOfEqualKeyID;
	PGPInt32				indexTry	= 0;
	PGPKeyDBObj *			keyObjTry;
	PGPKeyID				keyIDTry;
	int						comparison;

	pgpa((
		pgpaPGPKeySetValid(set),
		pgpaAddrValid(priv, EnumeratedSubsetPriv),
		pgpaAddrValid(key, char)));

	/* TBD assert object is key */

	/* Get the KeyID */
	pgpKeyID8( key, NULL, &keyID );

	/* Use binary search to find a key in the array with the same KeyID */
	lowerBound = 0;
	upperBoundPlusOne = priv->numKeys;
	while (lowerBound < upperBoundPlusOne)
	{
		indexTry = (lowerBound + upperBoundPlusOne) / 2;
		keyObjTry = priv->keyPointers[indexTry]->obj;
		if (keyObjTry == key)
			return priv->keyPointers[indexTry];	/* Found it! */

		pgpKeyID8( keyObjTry, NULL, &keyIDTry );
		comparison = PGPCompareKeyIDs(&keyID, &keyIDTry);
		if (comparison > 0)
			lowerBound = indexTry + 1;
		else if (comparison < 0)
			upperBoundPlusOne = indexTry;
		else
			break;
	}
	if (lowerBound >= upperBoundPlusOne)
		return NULL;	/* Binary search failed */
	
	/*
	 * We found a key with the same KeyID, but it's not the same key.
	 * Now we search the adjacent keys of equal KeyID looking for the key.
	 */
	indexOfEqualKeyID = indexTry;

	/* First search lower indices */
	while (--indexTry >= lowerBound)
	{
		keyObjTry = priv->keyPointers[indexTry]->obj;
		if (keyObjTry == key)
			return priv->keyPointers[indexTry];	/* Found it! */

		pgpKeyID8( keyObjTry, NULL, &keyIDTry );
		if (0 != PGPCompareKeyIDs(&keyID, &keyIDTry))
			break;
	}

	/* Then search higher indices */
	indexTry = indexOfEqualKeyID;
	while (++indexTry < upperBoundPlusOne)
	{
		keyObjTry = priv->keyPointers[indexTry]->obj;
		if (keyObjTry == key)
			return priv->keyPointers[indexTry];	/* Found it! */

		pgpKeyID8(keyObjTry, NULL, &keyIDTry);
		if (0 != PGPCompareKeyIDs(&keyID, &keyIDTry))
			break;
	}

	/* Now we know for sure it's not in the set */
	return NULL;
}


/*
 * Look for a child object below the specified key pointer.  Should only be
 * used if keyptr's children is not kAllChildren.  Returns the matched
 * child object, or its parent if it matched because of a kAllChildren at
 * the second level.  If object not found returns NULL.
 */
	static ObjEntry *
sFindChildOfKeyMember(
	ObjEntry *		keyptr,
	PGPKeyDBObj *	obj,
	PGPBoolean *	parentfound)
{
	PGPInt32		keychildren, keygrandchildren;
	ObjEntry *		objptr;

	keychildren = keyptr->numChildren;
	pgpAssert( keychildren != kAllChildren );

	objptr = keyptr + 1;

	while( keychildren-- > 0 ) {
		if (objptr->obj == obj) {
			*parentfound = FALSE;
			return objptr;
		}
		keygrandchildren = objptr->numChildren;
		if( keygrandchildren == kAllChildren ) {
			if( objptr->obj == obj->up ) {
				*parentfound = TRUE;
				return objptr;
			}
			++objptr;
		} else {
			++objptr;
			while( keygrandchildren-- > 0 ) {
				if( objptr->obj == obj ) {
					*parentfound = FALSE;
					return objptr;
				}
				++objptr;
			}
		}
	}
	*parentfound = FALSE;
	return NULL;
}


/*
 * Check to see if a keydbobj is a member of an enumerated subset
 */
	static PGPBoolean
IsMemberOfEnumeratedSubset(
	PGPKeySetRef			set,
	PGPKeyDBObjRef			obj)
{
	PGPKeyDBObj *			parent;
	ObjEntry *				keyptr;
	ObjEntry *				objptr;
	PGPBoolean				parentfound;

	if( PGPPeekKeyDBObjKeyDB( obj ) != set->keyDB )
		return FALSE;

	if( pgpObjectType( obj ) == RINGTYPE_KEY )
		return NULL != sFindKeyMemberOfEnumeratedSubset( set, obj );

	for( parent = obj->up; pgpObjectType( parent ) != RINGTYPE_KEY;
		 parent = parent->up )
		;

	keyptr = sFindKeyMemberOfEnumeratedSubset( set, parent );

	if( keyptr == NULL )
		return FALSE;

	if( keyptr->numChildren == kAllChildren )
		return TRUE;

	/*
	 * Here we have only a subset of the children in the set, so we must
	 * search for those.  Do a simple linear search since there usually
	 * aren't very many.
	 */

	objptr = sFindChildOfKeyMember( keyptr, obj, &parentfound );
	
	return objptr != NULL;
}

/* Check to see if the set is empty */
	static PGPBoolean
IsEmptyEnumeratedSubset(
	PGPKeySetRef			set )
{
	EnumeratedSubsetPriv *	priv = (EnumeratedSubsetPriv *)set->priv;
	return (priv->numKeys == 0);
}



/* Utility functions for unioning two enum keysets */

/* Copy key and all its children from given source to given offsets, update
 * offsets.  Return updated src pointer
 */
	static ObjEntry *
sCopyKeyAndChildren( ObjEntry *src, EnumeratedSubsetPriv *priv,
					 PGPUInt32 *objoff, PGPUInt32 *keyptroff )
{
	PGPInt32 numChildren;

	/* Copy first object */
	if( IsntNull( keyptroff ) )
		priv->keyPointers[(*keyptroff)++] = priv->objArray + *objoff;
	priv->objArray[(*objoff)].obj = src->obj;
	priv->objArray[(*objoff)++].numChildren = src->numChildren;

	/* Done if no children to copy */
	numChildren = src->numChildren;
	++src;
	if (numChildren == kAllChildren)
		return src;

	/* Else loop to copy children */
	while( numChildren-- > 0 )
		src = sCopyKeyAndChildren( src, priv, objoff, NULL );

	return src;
}

/* Merge the children of two identical keys.  Update offsets. */
/* Neither key has kAllChildren */
	static void
sMergeChildren( ObjEntry *key1, ObjEntry *key2, EnumeratedSubsetPriv *priv,
				PGPUInt32 *objoff, PGPUInt32 *keyptroff )
{
	PGPInt32 numChildren1, numChildren2;
	PGPInt32 numGrandchildren1, numGrandchildren2;
	PGPInt32 numChildren;
	ObjEntry *obj1, *obj2;
	PGPBoolean parentfound;
	PGPUInt32 childoff;

	pgpAssert( key1->numChildren != kAllChildren );
	pgpAssert( key2->numChildren != kAllChildren );

	obj1 = key1;
	obj2 = key2;

	priv->keyPointers[*keyptroff] = priv->objArray + *objoff;
	priv->objArray[(*objoff)++].obj = obj1->obj;

	numChildren1 = obj1->numChildren;
	numChildren2 = obj2->numChildren;
	numChildren = numChildren1;
	++obj1;

	while( numChildren1-- > 0 ) {
		obj2 = sFindChildOfKeyMember( key2, obj1->obj, &parentfound );
		/* Is this child in number 2? */
		if( IsNull( obj2 ) ) {
			/* Simple case, copy child and its children */
			obj1 = sCopyKeyAndChildren( obj1, priv, objoff, NULL );
		} else {
			/* parentfound is false else key2 would have had kAllChildren */
			pgpAssert( !parentfound );

			/* Now must check for merge at second level */
			/* Copy second level object */
			priv->objArray[(*objoff)].obj = obj1->obj;

			/* See if either one has kAllChildren */
			numGrandchildren1 = obj1->numChildren;
			numGrandchildren2 = obj2->numChildren;
			if( numGrandchildren1 == kAllChildren ) {
				priv->objArray[(*objoff)++].numChildren = kAllChildren;
				obj1++;
				if( numGrandchildren2 != kAllChildren )
					obj2 += numGrandchildren2;
				else
					obj2++;
			} else if( numGrandchildren2 == kAllChildren ) {
				priv->objArray[(*objoff)++].numChildren = kAllChildren;
				obj2++;
				obj1 += numGrandchildren1;
			} else {
				/* Must do item by item merge at third level */
				/* First copy all items from 1, then missing items from 2 */
				childoff = *objoff;
				++obj1;
				while( numGrandchildren1-- > 0 ) {
					priv->objArray[(*objoff)].obj = obj1->obj;
					priv->objArray[(*objoff)++].numChildren = 0;
					++obj1;
				}
				while( numGrandchildren2-- > 0 ) {
					++obj2;
					if( IsNull( sFindChildOfKeyMember( key1, obj2->obj,
													   &parentfound ) ) ) {
						/* Key2 grandchild not in key1, copy it */
						priv->objArray[(*objoff)].obj = obj2->obj;
						priv->objArray[(*objoff)++].numChildren = 0;
						++obj2;
					}
				}
				/* Update numChildren on childoff */
				priv->objArray[childoff].numChildren = (*objoff) - childoff;
			}
		}
	}

	/* Now copy remaining second level items from key2 that aren't in key1 */

	obj2 = key2 + 1;
	while( numChildren2-- > 0 ) {
		if( IsNull( sFindChildOfKeyMember( key1, obj2->obj, &parentfound ))) {
			obj2 = sCopyKeyAndChildren( obj2, priv, objoff, NULL );
			++numChildren;
		} else {
			/* Skip this, it has already been done */
			numGrandchildren2 = obj2->numChildren;
			if( numGrandchildren2 == kAllChildren )
				++obj2;
			else
				obj2 += numGrandchildren2;
		}
	}

	/* Update numChildren for top level merged key */
	priv->keyPointers[(*keyptroff)++]->numChildren = numChildren;
}


	static PGPError
MakeUnionOfEnumeratedSubsets(
	PGPKeySetRef		set1,
	PGPKeySetRef		set2,
	PGPKeySetRef *		newSet)
{
	PGPContextRef			context	= PGPPeekKeySetContext(set1);
	EnumeratedSubsetPriv *	priv1 = (EnumeratedSubsetPriv *)set1->priv;
	EnumeratedSubsetPriv *	priv2 = (EnumeratedSubsetPriv *)set2->priv;
	EnumeratedSubsetPriv *	priv;
	ObjEntry **				keyPtr1 = priv1->keyPointers;
	ObjEntry **				keyPtr2 = priv2->keyPointers;
	PGPKeyID				keyID1;
	PGPKeyID				keyID2;
	PGPKeySetRef			set;
	PGPUInt32				in1;
	PGPUInt32				in2;
	PGPUInt32				out;
	PGPUInt32				outptr;
	int						comparison;
	PGPError				err	= kPGPError_NoErr;

	*newSet = NULL;			/* In case there's an error */

	err = NewEnumeratedSubsetInternal(set1->keyDB, &set);
	if (IsPGPError(err))
		return err;

	priv = (EnumeratedSubsetPriv *)set->priv;
	priv->numKeys = priv1->numKeys + priv2->numKeys;
	priv->numObjs = priv1->numObjs + priv2->numObjs;

	priv->objArray = (ObjEntry *)pgpContextMemAlloc(context,
							priv->numObjs * sizeof(ObjEntry), 0);
	if (IsNull(priv->objArray))
	{
		PGPFreeKeySet(set);
		return kPGPError_OutOfMemory;
	}

	priv->keyPointers = (ObjEntry **)pgpContextMemAlloc(context,
							priv->numKeys * sizeof(ObjEntry *), 0);
	if (IsNull(priv->keyPointers))
	{
		PGPFreeKeySet(set);
		return kPGPError_OutOfMemory;
	}

	in1 = in2 = 0;

	out = 0;					/* Offset in ObjEntry array */
	outptr = 0;					/* Offset in keyPointer array */

	/* Merge the two sorted arrays until we reach the end of one of them */
	while (in1 < priv1->numKeys && in2 < priv2->numKeys)
	{
		/*
		 * Check for duplicate keys.  Note that if keyIDs aren't unique
		 * then this won't necessarily remove duplicates.  This isn't
		 * a bug, because duplicates needn't be removed for correctness.
		 * It is merely an optimization.
		 */
		if (keyPtr1[in1]->obj == keyPtr2[in2]->obj)
		{
			/* May need to merge child-objects here */
			if( keyPtr1[in1]->numChildren == kAllChildren)
			{
				/* No merge necessary, just copy one entry */
				sCopyKeyAndChildren( keyPtr1[in1], priv, &out, &outptr );
			} else if( keyPtr2[in2]->numChildren == kAllChildren ) {
				/* No merge necessary, just copy one entry */
				sCopyKeyAndChildren( keyPtr2[in2], priv, &out, &outptr );
			} else {
				/* Hard case, must do complicated merge */
				sMergeChildren( keyPtr1[in1], keyPtr2[in2], priv,
								&out, &outptr );
			}
			++in1; ++in2;
			continue;
		}

		pgpKeyID8( keyPtr1[in1]->obj, NULL, &keyID1);
		pgpKeyID8( keyPtr2[in2]->obj, NULL, &keyID2);

		comparison = PGPCompareKeyIDs(&keyID1, &keyID2);
		if (comparison <= 0) {
			sCopyKeyAndChildren( keyPtr1[in1++], priv, &out, &outptr );
		} else if (comparison > 0) {
			sCopyKeyAndChildren( keyPtr2[in2++], priv, &out, &outptr );
		}
	}

	/*
	 * Copy the rest of the remaining array.
	 * (at most one of these loops will be non-empty)
	 */
	while (in1 < priv1->numKeys)
		sCopyKeyAndChildren( keyPtr1[in1++], priv, &out, &outptr );
	while (in2 < priv2->numKeys)
		sCopyKeyAndChildren( keyPtr2[in2++], priv, &out, &outptr );

	/* Shrink the resulting array, in case there were duplicates */
	if (out < priv->numObjs)
		priv->numObjs = out;
	if( outptr < priv->numKeys )
		priv->numKeys = outptr;

	*newSet = set;
	return kPGPError_NoErr;
}

	static void
DestroyEnumeratedSubset(
	PGPKeySetRef	set)
{
	EnumeratedSubsetPriv *	priv = (EnumeratedSubsetPriv *)set->priv;
	PGPContextRef			context	= PGPPeekKeySetContext(set);

	if( IsntNull( priv ) )
	{
		if (IsntNull(priv->objArray))
		{
			pgpContextMemFree(context, priv->objArray);
			priv->objArray = NULL;
		}
		if (IsntNull(priv->keyPointers))
		{
			pgpContextMemFree(context, priv->keyPointers);
			priv->keyPointers = NULL;
		}
		priv->numObjs = 0;
		priv->numKeys = 0;
		pgpContextMemFree(context, priv);
	}
}

	static PGPError
NewEnumeratedSubsetInternal(
	PGPKeyDBRef			db,
	PGPKeySetRef *		newSet)
{
	PGPContextRef			context	= PGPPeekKeyDBContext(db);
	EnumeratedSubsetPriv *	priv;
	PGPKeySetRef			set;
	PGPError				err	= kPGPError_NoErr;

	*newSet = NULL;		/* In case there's an error */

	priv = (EnumeratedSubsetPriv *)pgpContextMemAlloc(context,
							sizeof(*priv), kPGPMemoryMgrFlags_Clear);
	if (IsNull(priv))
		return kPGPError_OutOfMemory;

	priv->keyPointers = NULL;
	priv->objArray = NULL;
	priv->numKeys = 0;
	priv->numObjs = 0;

	err = pgpNewKeySetInternal(db, &set);
	if (IsPGPError(err))
		return err;
	
	pgpAssertAddrValid(set, PGPKeySet);

	set->priv = priv;
	set->isMember = IsMemberOfEnumeratedSubset;
	set->isEmpty = IsEmptyEnumeratedSubset;
	set->makeUnion = MakeUnionOfEnumeratedSubsets;
	set->destroy = DestroyEnumeratedSubset;

	*newSet = set;
	return kPGPError_NoErr;
}

/*
 * Create an empty enumerated key set
 */
	PGPError
PGPNewEmptyKeySet(
	PGPKeyDBRef			keyDB,
	PGPKeySetRef *		newSet)
{
	PGPKeySetRef		set;
	PGPError			err	= kPGPError_NoErr;

	PGPValidatePtr( newSet );
	*newSet = NULL;	
	PGPValidateKeyDB( keyDB );

	pgpEnterPGPErrorFunction();

	err = NewEnumeratedSubsetInternal(keyDB, &set);
	if (IsPGPError(err))
		return err;

	*newSet = set;
	pgpAssertErrWithPtr( err, *newSet );
	return err;
}

/*
 * Create a singleton enumerated key set
 * If obj is not a key, singleton does not include its siblings or aunts.
 */
	static PGPError
pgpNewOneKeySetInternal(
	PGPKeyDBObjRef		obj,
	PGPKeySetRef *		newSet)
{
	PGPContextRef			context	= PGPPeekKeyDBObjContext(obj);
	EnumeratedSubsetPriv *	priv;
	PGPKeySetRef			set;
	PGPKeyDBObjRef			parent;
	PGPUInt32				numObjects;
	PGPError				err	= kPGPError_NoErr;

	err = NewEnumeratedSubsetInternal(PGPPeekKeyDBObjKeyDB(obj), &set);
	if (IsPGPError(err))
		return err;

	priv = (EnumeratedSubsetPriv *)set->priv;

	priv->keyPointers = (ObjEntry **)pgpContextMemAlloc(context,
											sizeof(ObjEntry *), 0);
	if (IsNull(priv->keyPointers))
	{
		PGPFreeKeySet(set);
		return kPGPError_OutOfMemory;
	}
	priv->numKeys = 1;

	/* Count how many objects we will have */
	numObjects = 1;
	for( parent = obj; pgpObjectType( parent ) != RINGTYPE_KEY;
												 parent = parent->up )
	{
		++numObjects;
	}
	pgpAssert( numObjects >= 1  &&  numObjects <= 3 );

	priv->objArray = (ObjEntry *)pgpContextMemAlloc(context,
										numObjects * sizeof(ObjEntry), 0);
	if (IsNull(priv->objArray))
	{
		PGPFreeKeySet(set);
		return kPGPError_OutOfMemory;
	}
	priv->numObjs = numObjects;
	priv->keyPointers[0] = priv->objArray;

	priv->objArray[0].obj = parent;
	priv->objArray[0].numChildren = (numObjects==1) ? kAllChildren : 1;

	if( numObjects > 1 ) {
		priv->objArray[1].obj = (numObjects==2) ? obj : obj->up;
		priv->objArray[1].numChildren = (numObjects==2) ? kAllChildren : 1;
	}
	if( numObjects > 2 ) {
		priv->objArray[2].obj = obj;
		priv->objArray[2].numChildren = 0;
	}
	*newSet = set;
	pgpAssertErrWithPtr( err, *newSet );
	return err;
}



	PGPError
PGPNewOneKeySet(
	PGPKeyDBObjRef		obj,
	PGPKeySetRef *		newSet)
{
	PGPKeySetRef		uncleSet;
	PGPUInt32			addtype;
	PGPKeyDBObj *		parobj;
	PGPKeyDBObj *		uncle;
	PGPError			err;

	PGPValidatePtr( newSet );
	*newSet = NULL;			/* In case there's an error */
	PGPValidateKeyDBObj( obj );

	pgpEnterPGPErrorFunction();

	err = pgpNewOneKeySetInternal( obj, newSet );
	if( IsPGPError( err ) )
		return err;

	/* Must add uncles - all subkeys if userid/sig, all userids if subkey */
	parobj = obj;
	while( !OBJISUSERID(parobj) && !OBJISKEY(parobj) )
		parobj = parobj->up;

	if( OBJISTOPKEY(parobj) )
		return kPGPError_NoErr;
	
	pgpAssert( OBJISTOPKEY( parobj->up ) );

	if( OBJISUSERID(parobj) )
		addtype = RINGTYPE_SUBKEY;
	else
		addtype = RINGTYPE_USERID;

	for( uncle = parobj->up->down; IsntNull(uncle); uncle = uncle->next )
	{
		if( !pgpKeyDBObjIsReal( uncle ) )
			continue;
		if( pgpObjectType( uncle ) != addtype )
			continue;
		/* Must add object */
		err = pgpNewOneKeySetInternal( uncle, &uncleSet );
		if( IsPGPError( err ) )
		{
			PGPFreeKeySet( *newSet );
			*newSet = NULL;
			return err;
		}
		PGPAddKeys( uncleSet, *newSet );
		PGPFreeKeySet( uncleSet );
	}

	return kPGPError_NoErr;
}


/*
 * Create an enum set from a list of key addresses.  We know these are all
 * keys, no sub objects, so we can simplify the code.  The key list is
 * already sorted by keyid.
 */
	PGPError
pgpNewKeyListSet(
	PGPKeyDB *			db,
	PGPKeyDBObjRef *	keylist,
	PGPUInt32			nkeys,
	PGPKeySetRef *		newSet)
{
	PGPContextRef			context;
	EnumeratedSubsetPriv *	priv;
	PGPKeySetRef			set;
	PGPUInt32				i;
	PGPError				err	= kPGPError_NoErr;

	PGPValidatePtr( newSet );
	*newSet = NULL;			/* In case there's an error */
	pgpAssert( nkeys > 0 );

	/* Use db from the list; sometimes we don't have a valid one otherwise */
	if( nkeys > 0 )
		db = PGPPeekKeyDBObjKeyDB( keylist[0] );

	context	= PGPPeekKeyDBContext(db);
	err = NewEnumeratedSubsetInternal(db, &set);
	if (IsPGPError(err))
		return err;

	priv = (EnumeratedSubsetPriv *)set->priv;

	priv->keyPointers = (ObjEntry **)pgpContextMemAlloc(context,
											nkeys * sizeof(ObjEntry *), 0);
	if (IsNull(priv->keyPointers))
	{
		PGPFreeKeySet(set);
		return kPGPError_OutOfMemory;
	}
	priv->numKeys = nkeys;

	priv->objArray = (ObjEntry *)pgpContextMemAlloc(context,
										nkeys * sizeof(ObjEntry), 0);
	if (IsNull(priv->objArray))
	{
		PGPFreeKeySet(set);
		return kPGPError_OutOfMemory;
	}
	priv->numObjs = nkeys;

	for( i=0; i<nkeys; ++i )
	{
		pgpAssert( pgpKeyIsValid( keylist[i] ) );
		pgpAssert( PGPPeekKeyDBObjKeyDB(keylist[i]) == db );
		priv->keyPointers[i] = &priv->objArray[i];
		priv->objArray[i].obj = keylist[i];
		priv->objArray[i].numChildren = kAllChildren;
	}
	*newSet = set;
	pgpAssertErrWithPtr( err, *newSet );
	return err;
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
